<?php
/******************************************************************************
* Catlair PHP Copyright (C) 2019  a@itserv.ru
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/



include_once ROOT."/datasource/datacore.php";
include_once ROOT."/datasource/data_factory.php";



class TMySQL extends TDataCore
{
    private $FResult = null;

    function __construct($ALog)
    {
        parent::__construct($ALog);
        $this->SetType(TDataFactory::MYSQL);
    }



    public function NativeToUniConverting($ANative)
    {
        switch ($ANative)
        {
            case 'BOOLEAN':     $Result = TDataCore::TYPE_BOOLEAN; break;

            case 'TINYINT':     $Result = TDataCore::TYPE_INT_8_SIGN; break;
            case 'SMALLINT':    $Result = TDataCore::TYPE_INT_16_SIGN; break;
            case 'MEDIUMINT':   $Result = TDataCore::TYPE_INT_24_SIGN; break;
            case 'INT':         $Result = TDataCore::TYPE_INT_32_SIGN; break;
            case 'BIGINT':      $Result = TDataCore::TYPE_INT_64_SIGN; break;

            case 'DATE':        $Result = TDataCore::TYPE_DATE; break;
            case 'TIME':        $Result = TDataCore::TYPE_TIME; break;
            case 'DATETIME':    $Result = TDataCore::TYPE_MOMENT; break;

            case 'CHAR':        $Result = TDataCore::TYPE_STRING; break;
            case 'VARCHAR':     $Result = TDataCore::TYPE_STRING_UNSIZE; break;

            case 'BINARY':      $Result = TDataCore::TYPE_BIN; break;
            case 'VARBINARY':   $Result = TDataCore::TYPE_BIN_UNSIZE; break;
            case 'LONGBLOB':    $Result = TDataCore::TYPE_BLOB; break;
            case 'JSON':        $Result = TDataCore::TYPE_JSON; break;

            case 'FLOAT':       $Result = TDataCore::TYPE_FLOAT; break;
            case 'DOUBLE':      $Result = TDataCore::TYPE_DOUBLE; break;

            case 'PRIMARY KEY': $Result = TDataCore::KEY_PRIMARY; break;
            case 'UNIQUE KEY':  $Result = TDataCore::KEY_UNIQUE; break;

            case 'NOT NULL':    $Result = TDataCore::NOT_EMPTY; break;

            default:            $Result = TDataCore::UNKNOWN; break;
        }
        return $Result;
    }



    public function UniToNativeConverting($AUni)
    {
        switch ($AUni)
        {
            case TDataCore::TYPE_BOOLEAN:       $Result = 'BOOLEAN'; break;

            case TDataCore::TYPE_INT_8_SIGN:    $Result = 'TINYINT'; break;
            case TDataCore::TYPE_INT_16_SIGN:   $Result = 'SMALLINT'; break;
            case TDataCore::TYPE_INT_24_SIGN:   $Result = 'MEDIUMINT'; break;
            case TDataCore::TYPE_INT_32_SIGN:   $Result = 'INT'; break;
            case TDataCore::TYPE_INT_64_SIGN:   $Result = 'BIGINT'; break;

            case TDataCore::TYPE_DATE:          $Result = 'DATE'; break;
            case TDataCore::TYPE_TIME:          $Result = 'TIME'; break;
            case TDataCore::TYPE_MOMENT:        $Result = 'DATETIME'; break;

            case TDataCore::TYPE_STRING:        $Result = 'CHAR'; break;
            case TDataCore::TYPE_STRING_UNSIZE: $Result = 'VARCHAR'; break;

            case TDataCore::TYPE_BIN:           $Result = 'BINARY'; break;
            case TDataCore::TYPE_BIN_UNSIZE:    $Result = 'VARBINARY'; break;
            case TDataCore::TYPE_BLOB:          $Result = 'LONGBLOB'; break;
            case TDataCore::TYPE_JSON:          $Result = 'JSON'; break;

            case TDataCore::TYPE_FLOAT:         $Result = 'FLOAT'; break;
            case TDataCore::TYPE_DOUBLE:        $Result = 'DOUBLE'; break;

            case TDataCore::KEY_PRIMARY:        $Result = 'PRIMARY KEY'; break;
            case TDataCore::KEY_UNIQUE:         $Result = 'UNIQUE KEY'; break;
            case TDataCore::NOT_EMPTY:          $Result = 'NOT NULL'; break;

            default:                    $Result = $AUni; break;
        }
        return $Result;
    }


    public function Opening()
    {
        @$this->SetHandle(mysqli_connect($this->GetHost(), $this->GetLogin(), $this->GetPassword()));

        if ($this->FHandle !== false)
        {
//            mysqli_set_charset($this->FHandle, 'utf8mb4');
            mysqli_select_db($this->FHandle, $this->GetDatabase());
            $this->SetResult(rcOk, '');
        }
        else
        {
            $this->SetResult(mysqli_connect_errno(), mysqli_connect_error());
        }
        return $this->FHandle !== false;
    }



    /*
     Disconnect from mysql
    */
    public function Closing()
    {
        $this->FLog->Begin();
        $this->FHandle = !mysqli_close($this->FHandle);
        $this->FLog->End();
        return $this->FHandle === false;
    }



    /*
    Return true if database $ADatabase:string exists
    */
    public function DatabaseExisting($ADatabase)
    {
        $this->SetCommand('show databases like "'.$ADatabase.'"');
        $this->Calling();
        return $this->GettingRecordCount() > 0;
    }




    /*
    Database $ADatabase:string create for mysql.
    Calling from TDatasource->DatabaseCreate.
    Return $this.
    */
    public function DatabaseCreating($ADatabase, $AParams)
    {
        $Cmd = 'create database '.$ADatabase;
        if ($AParams!==null)
        {
            if (array_key_exists('Codepage', $AParams)) $Cmd .= ' default charset '.$AParams['Codepage'];
        }
        $this->SetCommand($Cmd);
        $this->Calling();
        return $this;
    }



    /*
    Database $ADatabase:string deleting for mysql.
    Calling from TDatasource->DatabaseDelete.
    Return $this.
    */
    public function DatabaseDeleting($ADatabase)
    {
        $this->SetCommand('drop database '.$ADatabase);
        $this->Calling();
        return $this;
    }



    /*
    Selet database $ADatabase:string
    */
    public function DatabaseSelecting($ADatabase)
    {
        $this->SetCommand('use '.$ADatabase);
        $this->Calling();
        return $this;
    }



    /*
    Creating database administrator $ADatabase:string
    */
    public function DatabaseAdminCreating($ALogin, $APassword)
    {
        $FullLogin = '"'.$ALogin.'"@"'.$this->GetHost().'"';
        /* Create new user */
        $this->SetCommand('create user '.$FullLogin.' identified by "'.$APassword.'"');
        $this->Calling();
        /* Gift all priveleges for user */
        $this->SetCommand('GRANT ALL privileges on '.$this->GetDatabase().'.* to '.$FullLogin);
        $this->Calling();
        $this->SetCommand('FLUSH PRIVILEGES');
        $this->Calling();
        return $this;
    }




    /*
    Table creating
    $ATable
    $AFields
    $AIndexes
    */
    public function TableCreating($ATable, $AFields, $AIndexes)
    {
        $Fields = [];
        foreach ($AFields as $Field) array_push($Fields, $this->CmdFieldCreate($Field));

        /* Buil indexes command */
        $IndexList = [];
        if ($AIndexes!=null)
        {
            foreach ($AIndexes as $Index=>$IndexFields)
            {
                array_push($IndexList, ', INDEX idx_'.implode('_',$IndexFields).' ('.implode(', ',$IndexFields).')');
            }
        }

        $this->SetCommand($Cmd = 'CREATE TABLE '.$ATable.' ('.implode(', ', $Fields).' '.implode('', $IndexList).')');
        $this->Calling();
        return $this;
    }



    /*
    Table creating
    */
    public function &FieldCreating($ATable, $AField)
    {
        $Cmd = 'ALTER TABLE '.$ATable.' ADD '.$this->CmdFieldCreate($AField);
        $this -> SetCommand($Cmd);
        $this->Calling();
        return $this;
    }



    /*
    */
    public function UserExisting($ALogin, $AHost)
    {
        $this->SetCommand('select User from mysql.user where User="'.$ALogin.'" and Host="'.$AHost.'"');
        $this->Calling();
        return $this->GettingRecordCount()>0;
    }



    /*
    */
    public function &UserDeleting($ALogin, $AHost)
    {
        $this->SetCommand('drop user "' . $ALogin . '"@"'. $AHost .'"');
        $this->Calling();
        return $this;
    }



    /*
    Specific INSERT action for this Datasource
    */
    public function Inserting($ATableName, $AParams)
    {
        $Names = [];
        $Values = [];
        foreach ($AParams as $Name => $Value)
        {
            array_push($Names, (string)$Name);
            array_push($Values, $this->ValueToParam($Value));
        }
        $this->SetCommand('insert into ' . $ATableName . ' (' . implode(', ', $Names) . ') values (' . implode(', ', $Values) . ')');

        $this->Ending();
        $this->FResult = mysqli_query($this->FHandle, $this->GetCommand());

        $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
        return mysqli_insert_id($this->FHandle);
    }



    /*
    Specific UPDATE action for this Datasource
    */
    public function Updating($ATableName, $AConditions, $AValues)
    {
        $Values = [];
        foreach ($AValues as $Name => $Value)
        {
            $Line = $Name.'='.$this->ValueToParam($Value);
            array_push($Values, $Line);
        }
        $Command = 'UPDATE ' . $ATableName . ' SET ' . implode(', ', $Values) . $this->BuildWhere($AConditions);
        $this->SetCommand($Command);

        $this->Ending();

        /* MySQL do command*/
        $this->FResult = mysqli_query($this->FHandle, $this->GetCommand());

        /* Return result */
        $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
        return $this;
    }



    /*
    Call stored procedure for table $ATableName with input paremeters $AValue:array ["Name"=>"Value"]
    */
    public function Procing($ATableName, $AValues)
    {
        $this->Ending();

        $Values = [];
        foreach ($AValues as $Value)
        {
            $Line = $this->ValueToParam($Value);
            array_push($Values, $Line);
        }
        $this->SetCommand('call ' . $ATableName . '(' . implode(', ', $Values).')');
        /* MySQL do command*/
        $this->FResult = mysqli_query($this->FHandle, $this->GetCommand());
        /* Return result */
        $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));

        return $this;
    }



    /*
    Return next line as object
    */
    public function Reading()
    {
       $Result = mysqli_fetch_object($this->FResult);
       $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
       return $Result;
    }







    public function Deleting($ATableName, $AConditions)
    {
        /* Prepare */
        $Command = 'DELETE FROM ' . $ATableName . $this->BuildWhere($AConditions);;
        $this->SetCommand($Command);
        /* Execute command */
        $this->Ending();
        $Result = mysqli_query($this->FHandle, $this->GetCommand());
        /* Return result */
        $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
        return $this;
    }




    /*
    Specific SELECT action for this Datasource
    */
    public function &RecordSelecting($ATableName, $AFields, $AConditions)
    {
        /* Prepare SELECT query */
        $Command = 'SELECT ' . implode(', ',$AFields) . ' FROM ' . implode(', ',$ATableName) . $this->BuildWhere($AConditions);
        $this->SetCommand($Command);

        $this->Ending();

        /* send select */
        $this->FResult = mysqli_query($this->FHandle, $this->GetCommand());

        /* Return result */
        $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
        return $this;
    }




    /*
    Return true if database $ADatabase:string exists
    */
    public function GettingRecordCount()
    {
        if (gettype($this->FResult)!=='boolean') $Result = $this->FResult->num_rows;
        else $Result = 0;
        return $Result;
    }




    /*
    Specific Calling action for this Datasource
    */
    public function &Calling()
    {
       $this->Ending();
       /* MySQL do command*/
       $this->FResult = mysqli_query($this->FHandle, $this->GetCommand());
       $this->SetResult($this->GetSpecificResultCode(), mysqli_error($this->FHandle));
       return $this;
    }



    /*
    Finalize any interaction and datamanipulation with datasource.
    Can call after Select Delete Update Insert Exec and other.
    */
    public function &Ending()
    {
        while (mysqli_next_result($this->FHandle)) mysqli_store_result($this->FHandle);
        return $this;
    }




    /*
    */
    private function GetSpecificResultCode()
    {
        $Code = mysqli_errno($this->FHandle);
        if ($Code == '0') $Code = rcOk;
        else $Code = 'MySQL-'.$Code;
        return $Code;
    }



    /* Create Field params from array*/
    public function CmdFieldCreate($AField)
    {
        $Cmd = $AField['Name'].' ' . $this->UniToNative($AField['Type']);
        if (array_key_exists('Length', $AField)) $Cmd .= ' ('.$AField['Length'] .')';
        if (array_key_exists('Default', $AField)) $Cmd .= ' '.$this->UniToNative($AField['Default']);
        if (array_key_exists('Increment', $AField)) $Cmd .= ' '.$this->UniToNative($AField['Increment']);
        if (array_key_exists('Key', $AField)) $Cmd .= ' '.$this->UniToNative($AField['Key']);
        return $Cmd;
    }



    public function BinaryToString($ABinary)
    {
        return mysqli_real_escape_string($this->FHandle, $ABinary);
    }



    public function ValueToParam($AValue)
    {
        switch (gettype($AValue))
        {
            case 'boolean':
                if ($AValue) $Result = 'true';
                else $Result = 'false';
            break;
            case 'string':
                $Result = '"'.mysqli_real_escape_string($this->FHandle, $AValue).'"';
            break;
            case 'array':
                $Result = implode(';',$AValue);
            break;
            case 'integer':
            case 'double':
            case 'float':
                $Result = (string) $AValue;
            break;
            case 'NULL':
            default:
                $Result = 'null';
            break;
        }
        return $Result;
    }



    /* Build WHERE for operations from $AConditions
        Conditions in array Name=>Value
        Result Name1 = Value1 and Name2 = value2 and ...
    */
    private function BuildWhere($AConditions)
    {
        if ($AConditions === null) $Result = '';
        else
        {
            $Conditions = [];
            foreach ($AConditions as $Name => $Value) array_push($Conditions, $Name.'='.$this->ValueToParam($Value));
            $Result = ' WHERE ' . implode(' and ', $Conditions);
        }
        return $Result;
    }
}


