<?php
/**************************************************************************************
* Catlair PHP Copyright (C) 2019  a@itserv.ru
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <https://www.gnu.org/licenses/>.
*
**************************************************************************************
*
* Session class.
* Contain all information about current session
*
* Session files are keeping ROOT/catlair/session/[a]/[b]/[c]/[hash].json
*
* still@itserv.ru
*/



class TSession
{
    /* Private declration */

    private $FLog = null;
    private $JSON = null;
    public $Role = null;



    /**
    Constructor
    Create new session object. Must take $ACMS:TCMS
    */
    public function __construct($ACMS)
    {
        $this->FCMS = $ACMS;
    }




    public function &Open($ANew)
    {
        $SAPI = php_sapi_name();
        if ($ANew)
        {
            /* Create new session ID */
            do
            {
                $ID = md5(uniqid().uniqid());
            }
            while (file_exists($this->File($ID)));

            if ($this->JSON !== null)
            {
                $this->SetID($ID); /* Устанавливаем новый id */
                $this->Del('Login'); /* Удаляем из текущей сессии логин */
                $this->Del('LoginSite'); /* Удаляем из текущей сессии сайт логина */
                $this->Del('Token'); /* Удаляем из текущей сессии токен */
            }
        }
        else
        {
            $ID=null;
            /* Получение идентификатора из куки а если такового нет то по юзерагенту и адресу */
            if (array_key_exists('cl_session_id', $_COOKIE)) $ID = $_COOKIE['cl_session_id'];
            /* Create default ID by user data if session file not found */
            if ($ID==null || !file_exists($this->File($ID)))
            {
                switch ($SAPI)
                {
                    case 'cgi':
                    case 'fpm-fcgi':
                        $ID = md5($_SERVER['HTTP_HOST'] . $_SERVER['REMOTE_ADDR'] . $_SERVER['HTTP_USER_AGENT']);
                    break;
                    case 'cli':
                        $ID = 'localhost';
                    break;
                }
            }
            /* загрузка файла */
            $this->Load($ID);
        }



        /* Создание нового JSON в случае если файл не был загружен или ранее не был создан */
        if ($this->JSON === null)
        {
            /* создание нового массива */
            $this->JSON = [];
            /* установка параметров */
            switch ($SAPI)
            {
                case 'cgi':
                case 'fpm-fcgi':
                    $this->Set('Remote', $_SERVER['REMOTE_ADDR']);
                break;
                case 'cli':
                    $this->Set('Remote', 'localhost');
                break;
            }
            $this->SetLanguage(LANG_UNKNOWN);
            $this->SetSite(SITE_DEFAULT);
            $this->SetDomain(DOMAIN_UNKNOWN);
            $this->SetID($ID);
        }

        /* return session ID to cookies */
        setcookie('cl_session_id', $ID, 0);
        return $this;
    }



    /**
     * Завершение и закрытие сесси, те фактически удаление файла с диска.
     * После создается новая сессия.
     */
    public function Close()
    {
        if (unlink($this->File($this->GetID()))) $Result=rcOk;
        else $Result='ErrorCloseSession';
        if ($Result==rcOk) $this->Open(true);
        return $Result;
    }



    /*
     * Загрузить данные по сессии
     */
    private function &Load($AID)
    {
        // Читаем файл
        $File = $this->File($AID);
        if (file_exists($File))
        {
            /* Попытки чтения в цикле информации по сесиии. Цикл, для тех случаев когда файл занят */
            $JSONSource = '';
            do
            {
                $f = fopen($File, 'r');
                if ($f)
                {
                    $JSONSource = fread($f, 1024);
                    fclose($f);
                }
                else
                {
                    // усыпление для ожидания и что бы не грузить проц
                    usleep(10000);
                }
            } while ($JSONSource=='');

            /* JSON pars */
            $this->JSON = json_decode($JSONSource, true);
            if ($this->JSON!==null)
            {
                $this->SetID($AID);
                $this->Role = explode(';',$this->Get('Role',''));
            }
        }
        return $this;
    }



    /**
    Flush session data to file for next use.
    */
    public function Flush()
    {
        /* Get file name */
        $File = $this->File($this->GetID());
        $Path = pathinfo($File, PATHINFO_DIRNAME);

        /* Создание папки в которой будет хранится инфа по сессиями если ее нет */
        if (!file_exists($Path)) mkdir($Path, FILE_RIGHT, true);

        /* JSON convert to string */
        $JSONString = json_encode($this->JSON);

        /* Loop for write */
        $Result = false;
        for ($i=0; $i<10 && !$Result; $i++)
        {
            $Result = file_put_contents($File, $JSONString);
            if (!$Result) usleep(10000);
        };
        return $Result;
    }



    /**
    Проверка сессии является ли гостевой
    */
    public function IsGuest()
    {
        return $this->GetLogin()=='' && php_sapi_name()!=='cli';
    }



    private function Path()
    {
        return clRootPath().'/session';
    }



    private function File($AID)
    {
        return $this->Path() . '/' . clScatterName($AID, 3, 'UTF-8') . '.json';
    }



    /**
     * Чтение параметра дескрипта
     * $AKey - имя ключа
     * $ADefault - результат при отсутсвии ключа
     */
    public function Get($AKey, $ADefault)
    {
        if ($this->JSON!==null && $AKey!=null && array_key_exists($AKey, $this->JSON))
        {
            $Result = (string) $this->JSON[$AKey];
        }
        else $Result = $ADefault;
        return $Result;
    }



   /**
    * Запись параметра дескрипта
    * $AKey - имя ключа
    * $AValue - значение ключа
    */
    public function Set($AKey, $AValue)
    {
        $Result = rcUnknown;
        if ($this->JSON!==null && $AKey!=null)
        {
            $this->JSON[$AKey] = $AValue;
            $Result=rcOk;
        }
        return $Result;
    }



   /**
    * Запись параметра дескрипта
    * $AKey - имя ключа
    * $AValue - значение ключа
    */
    public function Del($AKey)
    {
        if ($this->JSON!==null && array_key_exists($AKey, $this->JSON)) unset($this->JSON[$AKey]);
        return $this;
    }



    /*
     * Work with named params
     */

    private function SetID($AID)
    {
        return $this->Set('ID',$AID);
    }



    public function GetID()
    {
        return (string) $this->Get('ID', null);
    }



    public function SetLanguage($AIDLang)
    {
        return $this->Set('Language', $AIDLang);
    }



    public function GetLanguage()
    {
        return (string)$this->Get('Language', LANG_UNKNOWN);
    }



    public function SetSite($ASite)
    {
        return $this->Set('Site', $ASite);
    }



    public function GetSite()
    {
        return (string) $this->Get('Site', null);
    }



    public function SetDomain($ADomain)
    {
        return $this->Set('Domain', $ADomain);
    }



    public function GetDomain()
    {
        return (string) $this->Get('Domain', null);
    }



    public function SetLogin($ALogin)
    {
        return $this->Set('Login', $ALogin);
    }



    public function GetLogin()
    {
        return (string) $this->Get('Login', null);
    }



    /*
    Очистка списка ролей
    */
    public function &RoleClear()
    {
        $this->Role = array ('guest');
        return $this;
    }



    /*
    Построение списка ролей
    */
    public function RoleBuild()
    {
        $From=new TDescript();
//        $From->Read($this->GetLogin());
        $From->ParentRead();
        $Param=null;
        $this->Role=array();
        $From->TraceParent('bind_secure', null, null, $Param, $this->Role);
        unset($From);

        foreach($this->Role as $Value) clDeb($Value);
        $this->JSON['Role'] = implode(';',$this->Role);
    }
}
