<?php
/**
Catlair PHP Copyright (C) 2019  a@itserv.ru

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

Catlair PHP Datasource module.

This modile contain base function for any databases. This module cannot create directly.
It has been created for child classes:

class TFileStorage extends TDatasource
class TMySQL extends TDatasource
class TMSSQL extends TDatasource
class TSQLight extends TDatasource
class TPostGreeSQL extends TDatasource

12.09.2019 still@itserv.ru
*/


/* Include catalir libraryes */
include_once(ROOT."/core/debug.php");
include_once(ROOT."/core/config.php");
include_once(ROOT."/datasource/data_factory.php");


class TDataCore
{
    /*Data Type*/
    const UNKNOWN =              'unknown';
    const TYPE_BOOLEAN =         'boolean';
    const TYPE_INT_8_SIGN =      'int_8_sign';
    const TYPE_INT_16 =          'int_16';
    const TYPE_INT_16_SIGN =     'int_16_sign';
    const TYPE_INT_24 =          'int_24';
    const TYPE_INT_24_SIGN =     'int_24_sign';
    const TYPE_INT_32 =          'int_32';
    const TYPE_INT_32_SIGN =     'int_32_sign';
    const TYPE_INT_64 =          'int_64';
    const TYPE_INT_64_SIGN=      'int_64_sign';
    const TYPE_MOMENT =          'moment';
    const TYPE_DATE =            'date';
    const TYPE_TIME =            'time';
    const TYPE_STRING =          'string';
    const TYPE_STRING_UNSIZE =   'string_unsize';
    const TYPE_BIN =             'binary';
    const TYPE_BIN_UNSIZE =      'binary_unsize';
    const TYPE_BLOB =            'blob';
    const TYPE_FLOAT =           'float';
    const TYPE_DOUBLE =          'double';
    const TYPE_JSON =            'json';

    /* Key type*/
    const KEY_UNIQUE =           'key_unique';
    const KEY_PRIMARY =          'key_primary';

    /* Key type*/
    const NOT_EMPTY =            'not_empty';




    /* Protected declaraions */
    protected $FLog = null;             /* TLog from debut.php */
    protected $FHandle = false;         /* Handle for any object as File, MySQL, MSSQL etc */

    /* Private declarations */
    private $FLogin = '';               /* Login */
    private $FPassword = '';            /* Password */
    private $FHost = '';                /* Host */
    private $FDatabase = '';            /* Database */
    private $FOpened = false;           /* Connected boolan true or false */
    private $FResultCode = rcUnknown;   /* Last result code */
    private $FResultMessage = '';       /* Last result message */
    private $FResultID = false;         /* Last result ID for Insert poeration */
    private $FCommand = '';             /* Last command text after Prepare* functions */
    private $FType = '';                /* Type datasource. It property set in child specific classes */



    /*
    Constructor receive $ALog:TLog
    */
    function __construct($ALog)
    {
        $this->FLog = $ALog;
        $this->FType = TDataFactory::DEFAULT;
    }


    /**
    Datasource
    */


    /*
    Open datasource
    */
    public function &Open()
    {
        if (!$this->FOpened && $this->CheckMethod('Opening'))
        {
            $this->FLog->Begin();
            $this->FOpened = $this->Opening();
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Close datasource
    */
    public function Close()
    {
        if ($this->CheckOpened() && $this->CheckMethod('Closing'))
        {
            $this->FLog->Begin();
            $this->FOpened = !$this->Closing();
            $this->FLog->End();
        }
        return $this;
    }



    public function Call()
    {
        if ($this->CheckOpened() && $this->CheckMethod('Opening'))
        {
            $this->FLog->Begin();
            $this->Calling();
            $this->FLog->End();
        }
        return $this;
    }



    public function Proc($AProcName, $AParameters)
    {
        if ($this->CheckOpened() && $this->CheckMethod('Opening'))
        {
            $this->FLog->Begin();
            $this->Procing($AProcName, $AParameters);
            $this->FLog->End();
        }
        return $this;
    }



    /**************************************************************************
    Databse
    */

    /*
    Check database $ADatabase:string exists and return true or false
    */
    public function DatabaseExists($ADatabase)
    {
        $Result = false;
        if ($this->CheckOpened() && $this->CheckMethod('DatabaseExisting'))
        {
            $this->FLog->Begin();
            $Result = $this->DatabaseExisting($ADatabase);
            $this->FLog->End();
        }
        return $Result;
    }



    /*
    Create database $ADatabase:string.
    */
    public function &DatabaseCreate($ADatabase, $AParams)
    {
        if ($this->CheckOpened() && $this->CheckMethod('DatabaseCreating'))
        {
            $this->FLog->Begin();
            $this->DatabaseCreating($ADatabase, $AParams);
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Create database $ADatabase:string.
    */
    public function &DatabaseDelete($ADatabase)
    {
        if ($this->CheckOpened() && $this->CheckMethod('DatabaseDeleting'))
        {
            $this->FLog->Begin();
            $this->DatabaseDeleting($ADatabase);
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Select database $ADatabase:string.
    */
    public function &DatabaseSelect()
    {
        if ($this->CheckOpened() && $this->CheckMethod('DatabaseSelecting'))
        {
            $this->FLog->Begin();
            $this->DatabaseSelecting($this->FDatabase);
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Create database administrator
    */
    public function &DatabaseAdminCreate($ALogin, $APassword)
    {
        if ($this->CheckOpened() && $this->CheckMethod('DatabaseAdminCreating'))
        {
            $this->FLog->Begin();
            $this->DatabaseAdminCreating($ALogin, $APassword);
            $this->FLog->End();
        }
        return $this;
    }


    /*
    Users
    */


    /*
    Check user exists by $ALogin and $Host
    */
    public function UserExists($ALogin, $Host)
    {
        $this->FLog->Begin();
        if ($this->CheckOpened()) $Result = $this->UserExisting($ALogin, $Host);
        else $Result=false;
        $this->FLog->End();
        return $Result;
    }



    /*
    User delete
    */
    public function UserDelete($ALogin, $Host)
    {
        $this->FLog->Begin();
        if ($this->CheckOpened()) $this->UserDeleting($ALogin, $Host);
        $this->FLog->End();
        return $this;
    }



    /**************************************************************************
    Table
    */


    /*
    Create Table
    */
    public function &TableCreate($AName, $AFields, $AIndexes)
    {
        if ($this->CheckOpened() && $this->CheckMethod('TableCreating'))
        {
            $this->FLog->Begin();
            $this->TableCreating($AName, $AFields, $AIndexes);
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Field create
    */
    public function &FieldCreate($ATableName, $AField)
    {
        if ($this->CheckOpened() && $this->CheckMethod('FieldCreating'))
        {
            $this->FLog->Begin();
            $this->FieldCreating($ATableName, $AField);
            $this->FLog->End();
        }
        return $this;
    }



    /**************************************************************************
    Records
    */


    /*
    Insert new record in to $ATableName:string from $AParams:array['Name'=>'Value'...]
    */
    public function RecordInsert($ATableName, $AParams)
    {
        $this->FLog->Begin();
        if ($this->CheckOpened()) $this->FResultID = $this->Inserting($ATableName, $AParams);
        $this->FLog->End();
        return $this;
    }



    /*
    Update record in to $ATableName:string from $AParams:array['Name'=>'Value'...]
    */
    public function RecordUpdate($ATableName, $AConditions, $AValues)
    {
        $this->FLog->Begin();
        if ($this->CheckOpened()) $this->Updating($ATableName, $AConditions, $AValues);
        $this->FLog->End();
        return $this;
    }




    /*
    Delete exists cortege in $ATableName:string
    */
    public function RecordDelete($ATableName, $ACondition)
    {
        $this->FLog->Begin();
        if ($this->CheckOpened()) $this->Deleting($ATableName, $ACondition);
        $this->FLog->End();
        return $this;
    }



    /*
    Select record from table $ATableName:string and contitions $AParams:array['Name']
    */
    public function RecordSelect($ATableName, $AFields, $ACondition)
    {
        if ($this->CheckOpened())
        {
            $this->FLog->Begin();
            $this->RecordSelecting($ATableName, $AFields, $ACondition);
            $this->FLog->End();
        }
        return $this;
    }



    /*
    Select record from table $ATableName:string and contitions $AParams:array['Name']
    */
    public function RecordSelectByField($ATableName, $AFields, $ACondition)
    {
        if ($this->CheckOpened())
        {
            $this->FLog->Begin();
            $this->RecordSelectingByField($ATableName, $AFields, $ACondition);
            $this->FLog->End();
        }
        return $this;
    }





    public function NativeToUni($AValue)
    {
       return $this->NativeToUniConverting($AValue);
    }



    public function UniToNative($AValue)
    {
       return $this->UniToNativeConverting($AValue);
    }



    /*
    Return safe $AValue:string for files.
    This procedure prevents SQL enjections. All values for SQL params must be processed by this method.
    */
    public function SafeValue($AValue)
    {
        return str_replace('"', "'", $AValue);
    }



    /*
    Set last result code and message
    */
    public function SetResult($ACode, $AMessage)
    {
        $this->FResultCode = $ACode;
        $this->FResultMessage = $AMessage;
        if ($ACode != rcOk)
        {
            $this->FLog->Error()->Param('SQL error code', $ACode)->Text($AMessage);
            $this->FLog->Debug()->Param('Database',$this->GetDatabase())->Text($this->FCommand);
        }
        return $this;
    }



    /*
    Set last result code and message
    */
    public function IsOk()
    {
        return $this->FResultCode==rcOk;
    }



    /*
    Return last result code
    */
    public function GetResultMessage()
    {
        return $this->FResultMessage;
    }



    /*
    Return last result code
    */
    public function GetResultCode()
    {
        return $this->FResultCode;
    }



    /* Read next result */
    public function Read()
    {
        return $this->Reading();
    }


    /*
    Finalize any interaction and datamanipulation with datasource.
    Can call after Select Delete Update Insert Exec and other.
    */
    public function &End()
    {
        $this->FLog->Begin();
        $this->Ending();
        $this->FLog->End();
        return $this;
    }






    /**************************************************************************
    Prepare commands for datasourse By default use DML for MySQL
    */


    /*
    Property getters and setters
    */

    public function &SetOpened($AValue)
    {
        if ($AValue)
        {
            if (!$this->FOpened) $this->Open();
            else $this->Close();
        }
        return $this;
    }



    public function GetResultID()
    {
        return $this->FResultID;
    }



    public function SetResultID($AValue)
    {
        $this->FResultID = $AValue;
        return $this;
    }



    /*
    Return current login
    */
    public function GetOpened()
    {
        return $this->FOpened;
    }



    /*
    Set handle
    */
    public function &SetHandle($AValue)
    {
        $this->FHandle = $AValue;
        return $this;
    }



    /*
    Return last command after Prepare*
    */
    public function GetCommand()
    {
        return $this->FCommand;
    }



    /*
    Set command for Insert Update Select and other...
    */
    public function &SetCommand($AValue)
    {
/*        $this->FLog->Debug()->Text($AValue);*/
        $this->FCommand = $AValue;
        return $this;
    }



    /*
    Set command for Insert Update Select and other...
    */
    public function &LoadCommand($AFileName)
    {
        if (file_exists($AFileName)) $this->SetCommand(file_get_contents($AFileName));
        else $this->FLog->Warning()->Param('File not found', $AFileName);
        return $this;
    }


    /*
    Set command for Insert Update Select and other...
    */
    public function GetRecordCount()
    {
        return $this->GettingRecordCount();
    }


    /**
    Connection getters & setters
    */

    /*
    Return current login
    */
    public function GetLogin()
    {
        return $this->FLogin;
    }



    /*
    Set login form $AValue:string
    */
    public function SetLogin($AValue)
    {
        $this->FLogin = $AValue;
        return $this;
    }



    /*
    Return password
    */
    public function GetPassword()
    {
        return $this->FPassword;
    }



    /*
    Set password from $AValue:string
    */
    public function SetPassword($AValue)
    {
        $this->FPassword = $AValue;
        return $this;
    }



    public function GetHost()
    {
        return $this->FHost;
    }



    public function SetHost($AValue)
    {
        $this->FHost = $AValue;
        return $this;
    }



    public function GetType()
    {
        return $this->FType;
    }



    public function SetType($AValue)
    {
        $this->FType = $AValue;
        return $this;
    }



    public function GetDatabase()
    {
        return $this->FDatabase;
    }



    /*
    Set database for current.
    This do not change database.
    */
    public function SetDatabase($AValue)
    {
        $this->FDatabase = $AValue;
        return $this;
    }



    public function &GetLog()
    {
        return $this->FLog;
    }



    public function &SetLog($AValue)
    {
        $this->FLog = $AValue;
        return $this;
    }



    private function CheckOpened()
    {
        if (!$this->FOpened)
        {
            $this->FLog->Warning()->Text('Datasource is close');
            $this->FResultID = false;
        }
        return $this->FOpened;
    }



    private function CheckMethod($AName)
    {
        if (!method_exists($this, $AName))
        {
            $this->FLog->Warning()->Param('Method not found', $AName);
            $Result = false;
        }
        else $Result = true;
        return $Result;
    }



    public function &StoreConfig($AFileName)
    {
        $Config = new TConfig();
        $Config->SetFile($AFileName);
        $Config->GetParams()->Type = $this->FType;
        $Config->GetParams()->Login = $this->FLogin;
        $Config->GetParams()->Password = $this->FPassword;
        $Config->GetParams()->Host = $this->FHost;
        $Config->GetParams()->Database = $this->FDatabase;
        $Config->Flush();
        return $this;
    }



    public function &RestoreConfig($AFileName)
    {
        $Config = new TConfig();
        $Params = $Config->SetFile($FileName)->Read()->GetParams();

        $this->FLogin = $Params->Login;
        $this->FPassword  = $Params->Password;
        $this->FHost = $Params->Host;
        $this->FDatabase = $Params->Database;
        /* Write JSON to file */
        if (file_put_contents($AFileName, json_encode($JSON))) $this->SetResult(rcOk,'');
        else
        {
            $this->SetResult('ErrorStoreDatabaseConfig', $AFileName);
            $this->FLog->Warning()->Text($this->GetErrorCode)->Param('File',$AFileName);
        }
        /* Return result */
        return $this;
    }



    private function BinaryToString($ABinary)
    {
        return bin2hex($ABinary);
    }
}
