<?php
/**
* Catlair PHP Copyright (C) 2019  a@itserv.ru
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <https://www.gnu.org/licenses/>.
*
* Class for compiling content with embeded XML constructions:
* <cl param="command" param="value" ... />
* or
* <cl param="command" param="value" ... >
*   <command param="value"/>
* </cl>
*
* still@itserv.ru
*
*/

include_once "debug.php"; /* Include debug system */
include_once "utils.php"; /* Include base utils for project */
include_once "controller.php"; /* Include controller for API calls */

class TBuilder extends TLog
{
    /* Settings */
    private $FContent = '';         /* Current content */
    private $FRecursDepth = 100;    /* Maximum recursion depth */
    private $RootPath = '.';        /* Root path for content and libraries */

    /* Search and replace arrays */
    private $FSearch = [];
    private $FReplace = [];

    /* Internal objects */
    private $FOnUnknownCommand = null;
    private $FOnLibraryPath = null;


    /*
     * Set content
     */
    public function &SetContent($AContent)
    {
        $this->FContent = $AContent;
        return $this;
    }



    /*
     * Get content
     */
    public function GetContent()
    {
        return $this->FContent;
    }



    /*
     * Set depth of recursion from $ARecursDepth:integer
     */
    public function &SetRecursDepth($ARecursDepth)
    {
        $this->FRecursDepth = $ARecursDepth;
        return $this;
    }



    /*
     * Get depth of recursion
     */
    public function GetRecursDepth()
    {
        return $this->FRecursDepth;
    }



    /*
     * Set Root path from $APath:string for file operations.
     */
    public function &SetRootPath($APath)
    {
        $this->FRootPath = $APath;
        return $this;
    }



    /*
     * Return root path for all file operations.
     */
    public function &GetRootPath()
    {
        return $this->FRootPath;
    }



    /*
     * Building content
     */
    public function &Build()
    {
        $this->FContent = $this->Parsing($this->FContent);
        $this->FContent = $this->Replace($this->FContent);
        return $this;
    }



    /*
     * Send content to browser
     */
    public function &Send()
    {
        print($this->FContent);
        return $this;
    }



    /*
     * Parsing any content from $AContent:string and return result
     */
    public function Parsing($AContent)
    {
        return $this->Pars($AContent, 0);
    }



    /*
     * Create new key with $AName:string and $AValue:any for Replace operations
     */
    public function &AddKey($AName, $AValue)
    {
        array_push($this->FSearch, $AName);
        array_push($this->FReplace, $AValue);
        return $this;
    }



    /*
     * Replace all keys in the content $AContent:string and return it
     */
    private function Replace($AContent)
    {
        return str_replace($this->FSearch, $this->FReplace, $AContent);
    }



    /*
     * Recurcive parsing for content from $ACountent:string with depth $ADepth:integer
     * After parsing funciton will return new content
     */
    private function Pars($AContent, $ADepth)
    {
        $ADepth = $ADepth + 1;
        if ( $ADepth < $this->FRecursDepth )
        {
            do
            {
                /* Getting list of tags over regvar with cl tag */
                preg_match('/\<cl(?:(\<)|".+?"|.|\n)*?(?(1)\/cl|\/)\>/', $AContent, $m, PREG_OFFSET_CAPTURE);
                if (count($m) > 0)
                {
                    $b = $m[0][1];
                    $l = strlen ($m[0][0]);
                    $Source = $m[0][0];
                    if ( $l > 0 )
                    {
                        $Source = $this->Replace($Source);
                        $XMLSource = '<?xml version="1.0"?>' . $Source;
                        $XML = @simplexml_load_string ($XMLSource);
                        if ( $XML )
                        {
                            $Content = '';
                            $this->BuildElement($Content, $XML, $ADepth);
                            $Result=$Content;
                        }
                        else $Result = $this->HTMLError('XMLError', 'Error XML pars', $XMLSource);
                        /* Check recurstion error */
                        if ( $ADepth+1 ==  $this->FRecursDepth ) $Result .= $this->Error('Recurs', 'Recursion depth limit '.$ADepth, $XMLSource);
                        $AContent = trim(substr_replace ($AContent, $Result, $b, $l));
                    }
                }
            } while ( count($m)>0 );
        } else $AContent='';

        return $AContent;
    }



    /*
     * Resursive processing of XML command in cl tag
     * $AContent:string - contain text content
     * $AElement:simplexml - it is text <cl param="value".../> or multiline XML <cl><command params="value"...></cl>.
     * $ARecursDepth:integer - it is a current recursion depth. Zero by default.
     */
    private function BuildElement(&$AContent, &$AElement, $ARecursDepth)
    {
        /* Processing of string as a key with parametes */
        $this->Exec($AContent, $AElement->getName(), null, $AElement);
        /* Processing of directives in pair key=>value */
        if ($AElement->getName() =='cl') foreach ($AElement->attributes() as $Key => $Value) $this->Exec($AContent, $Key, $Value, $AElement);
        /* Processing of internal cl tags */
        $AContent = $this->Pars($AContent, $ARecursDepth);
        /* Prpcessing of child stings with keys */
        foreach ($AElement -> children() as $Line => $Param) $this->BuildElement($AContent, $Param, $ARecursDepth);
        return $this;
    }


    /*
     * Main work with cl tag $AContent:string content
     */
    private function Exec(&$AContent, $Command, $AValue, $Params)
    {
        $r = array('NoError'=>false, 'Message'=>'', 'Error'=>'');

        switch (strtolower($Command))
        {

            default:
                if ($AValue) $AContent = str_replace('%'.$Command.'%', $AValue, $AContent);
                else
                {
                    /* Unknow key is found and it is not a macrochange string (%example%) in cl tag */
                    $r= ['Error'=>'UnknownKey', 'Message' => 'Unknown key ['.$Command.'] (cl; set; add; file; replace; convert; exec; header; include ets...)'];
                }
            break;


            /* Empty tag. Nothing to do */
            case 'cl':
            break;


            /* Involuntary parsing for last content */
            /* <cl .... pars="true"/> */
            case 'pars':
                if ($AValue) $Value = $AValue;
                else $Value = (string) $Params['value'];
                if ($Value == 'true') $AContent = $this->Pars($AContent, 0);
            break;


            /* Set new content from $AValue:string */
            /* <cl content="IDContent"/> */
            case 'set':
                if ($AValue) $AContent = $AValue;
                else
                {
                    if ($Params['value']) $AContent = $Params['value'];
                    else
                    {
                        $r['Error'] = 'ParamNotFound';
                        $r['Message'] = 'Parameter <b>value</b> not found';
                    }
                }
            break;


            /* Adding of content from $AValue:string */
            /* <cl ... add="IDContent"/> */
            case 'add':
                if ($AValue) $Value = $AValue;
                else $Value = (string) $Params['value'];
                $AContent =  $AContent .= $Value;
            break;


            /* This is an uncompleted URL builder. I must do it. */
            case 'url':
                $Search = [];
                $Replace = [];
                /* параметры из */
                foreach ($clURL as $Key => $Value)
                {
                    array_push($Search, '%'.$Key.'%');
                    array_push($Replace, $Value);
                }
                $AContent = str_replace($Search, $Replace, $AContent);
            break;


            /* Replace one parameter */
            case 'replace':
                $AContent = str_replace($Params['from'], $Params['to'], $AContent);
            break;


            /* Mass replace parameters */
            case 'masreplace':
                $Search = array();
                $Replace = array ();
                foreach ($Params->attributes() as $Key => $Value)
                {
                    array_push($Search, '%'.$Key.'%');
                    array_push($Replace, $Value);
                }
                $AContent = str_replace($Search, $Replace, $AContent);
            break;


            /* Optimize content */
            case 'optimize':
            case 'pure':
                $AContent = preg_replace('/  +/','', preg_replace('/[\r\n]/',' ',$AContent));
            break;


            /* Convert content to clear, html, pure, uri, md5 */
            case 'convert':
                if ($AValue) $To = $AValue;
                else $To = $Params['to'];
                $To=strtolower($To);
                switch ($To)
                {
                    case 'clear': $AContent = ''; break;
                    case 'html': $AContent = htmlspecialchars($AContent); break;
                    case 'pure': $AContent = preg_replace('/  +/','', preg_replace('/[\r\n]/',' ',$AContent)); break;
                    case 'uri': $AContent = encodeURIComponent ($AContent); break;
                    case 'md5': $AContent = md5 ($AContent); break;
                    case 'default':; break;
                    default:
                        $r['Error']='UnknownConvert';
                        $r['Message']='Unknown convert mode ['.$To.'] (clear; html; pure; uri; md5)';
                    break;
                }
            break;


            /* Get content from file or descript */
            case 'file':
            case 'content':
                $this->BeginLabel('Get content');
                if ($AValue) $ID = (string) $AValue;
                else $ID = (string) $Params['id'];

                if ($ID!='none')
                {
                    $r = $this->ExtContent($ID);
                    if (!$r['Error']) $AContent = $r['Content'];
                }
                $this->End();
            break;


            /* Include library */
            case 'library':
            case 'include':
                $this->BeginLabel('Include library');

                if ($AValue) $PrefixLib = (string)$AValue;
                else $PrefixLib = (string)$Params['name'];

                $this->Debug()->Param('Library',$PrefixLib);

                /* Call upper level */
                $r = $this->ExtIncludePath($PrefixLib, $Command);
                if (!$r['Error'] && $r['Continue']==true)
                {
                    $FileName = $r['Path'];
                    if (file_exists($FileName))
                    {
                        /* Load library */
                        $this->Debug()->Param('Library path',$FileName);
                        include_once ($FileName);
                    }
                    else
                    {
                        /* Error load library */
                        $r = ['Error'=>'UnknownLibrary', 'Message'=>'PHP library not found in ['.$FileName.']'];
                    }
                }

                $this->End();
            break;


            /* Execute PHP function */
            /* This is an old method and is not recommended */
            case 'exec':
                $this->BeginLabel('Execute user function');

                if ($AValue) $NameFunc = $AValue;
                else $NameFunc = $Params['name'];

                /* Call extended function */
                $r = $this->ExtFunctionName($NameFunc);

                /**/
                if (!$r['Error'])
                {
                    $NameFunc=$r['Name'];
                    $this->Debug()->Param('Funciton name', (string) $NameFunc);

                    if (function_exists($NameFunc))
                    {
                        $Controller = new TController($this);
                        $Controller -> SetContent($AContent);
                        call_user_func_array($NameFunc,  array($Params, $Controller, $this));
                        $AContent = $Controller->End();
                        unset($Controller);
                    }
                    else
                    {
                        $r['Error'] = 'UnknownFunction';
                        $r['Message'] = 'Function [' . $NameFunc . '] not found.';
                    }
                }

                $this->End();
            break;



            /* Execute PHP code over controller */
            /* It is recommended instead of exec */
            case 'call':
                $this->BeginLabel('Call controller');

                if ($AValue) $Call = (string)$AValue;
                else $Call = (string)$Params['name'];

                $this->LoadController($Call, $r);
                if ($r['Error']=='')
                {
                    $Controller = new $r['Class']($this);
                    $Controller->SetContent($AContent);
                    foreach ($Params->attributes() as $Key => $Value) $Controller->SetIncome((string)$Key, (string)$Value);
                    $Controller->SetTypeContentIncome();
                    call_user_func(array($Controller, $r['Method']));
                    $AContent = $Controller->End();
                    unset($Controller);
                }

                $this->End();
            break;



            /* Write HTTP header */
            case 'header':
                /* Collect params */
                if ($AValue) $St = (string)$AValue;
                else $St = (string)$Params['value'];
                /* Work */
                if (PHP_SAPI !== 'cli') header($St);
                else $this->Warning()->Param('Header is not applyed on cli mode', $St);
            break;



            /* Set redirect for FPM page */
            case 'redirect':
                /* Collect params */
                if ($AValue) $URL = $AValue;
                else $URL = $Params['url'];
                /* Work */
                if (PHP_SAPI !== 'cli')
                {
                    if ($URL) header('Location: '.$URL);
                    else
                    {
                        $r['Error']='ParamererNoFound';
                        $r['Message']='Parameter <b>url</b> not found';
                    }
                }
            break;


            /* Suppression of errors */
            case 'error':
                /* Collect params */
                if ($AValue) $Value = $AValue;
                else $Value = $Params['value'];
                /* Work */
                if (strtolower($Value) == 'false')
                {
                    $r['Error'] = '';
                    $r['Message'] = '';
                }
            break;
        }


        if ($r['Error'])
        {
            $Line = $Params->getName();
            foreach ($Params->attributes() as $Key => $Value) $Line.= ' ' . $Key . '="' . $Value . '"';
            $AContent .= $this->HTMLError($r['Error'], $r['Message'], '<'.$Line.'/>');
        }

        return $this;
    }



    /* Return formated error line in HTML */
    public function HTMLError($ACode, $AMessage, $ASource)
    {
        return '<div class="Error"><div class="Code">'.$ACode.'</div><div class="Message">'.$AMessage.'</div><div class="Source">'.htmlspecialchars($ASource).'</div></div>';
    }



    /* Return path and file from root */
    private function GetFileNameFromRoot ($AFileName)
    {
        return clPathControl($this->RootPath.'/'.$AFileName);
    }


    /* Load controller from $ACall:string in format "ControllerName.MethodName" */
    public function &LoadController($ACall, &$r)
    {
        /*Get class name and method name*/
        $Split = explode('.',$ACall);
        if (count($Split)!=2) $r = ['Error' => 'CallFormatError', 'Message' => 'Call can consist "class.method"'];
        else
        {
            $Class = $Split[0];
            $this->BeginLabel('Include controller library');
            /* Call upper level */
            $r = $this->ExtIncludePath($Class, 'controller');
            if (!$r['Error'] && $r['Continue']==true)
            {
                $FileName = $r['Path'];
                if (file_exists($FileName))
                {
                    /* Load library */
                    $this->Debug()->Param('Library path', $FileName);
                    include_once ($FileName);
                }
                else
                {
                    /* Loading library error */
                    $r = ['Error'=>'UnknownLibrary', 'Message'=>'PHP library not found in ['.$FileName.']'];
                    $this->Warning()->Param('Unknown library', $FileName);
                }
            }

            /* Controll class */
            if (class_exists($Class))
            {
                $Method = $Split[1];
                if (method_exists($Class, $Method))
                {
                    $r = ['Error'=>'', 'Class'=>$Class,'Method'=>$Method];
                }
                else
                {
                    $r = ['Error' => 'UnknownMethod', 'Message' => 'Unknown method ['.$Method.'] for class ['.$Class.']'];
                    $this->Warning()->Param('Unknown method', $Method);
                }
            }
            else
            {
                $r = ['Error' => 'UnknownClass', 'Message' => 'Unknown class'];
                $this->Warning()->Param('Unknown class', $Class);
            }
            $this->End();
        }
        return $this;
    }




    /*
     * Read end return file content by $AID:string
     * Warning! This function is not safe, because it returns any file from RootPath without controling path.
     * This function must be overridden in upper level.
     * Use this function as a template only.
     */
    public function ExtContent($AID)
    {
        /* Get file name */
        $FileName=$this->GetFileNameFromRoot($AID);

        if (file_exists($FileName))
        {
            $Content = @file_get_contents($FileName);
            $Result = ['Content'=>$Content, 'Error'=>'', 'Message'=>''];
        }
        else
        {
            $this->Error()->Param('File not found', $FileName);
            $Result = ['Content'=>'', 'Error'=>'FileNotFound', 'Message'=>'File ['.$FileName.'] not found.'];
        }
        return $Result;
    }



    /*
    Extended function for including libraries
     */
    public function ExtIncludePath($ALibrary, $ACommand)
    {
        return ['Path'=>$ALibrary, 'Error'=>'', 'Message'=>''];
    }



    /*
    Extended function for extended call when need function name for exec
    */
    public function ExtFunctionName($AFunction)
    {
        return ['Name'=>$AFunction, 'Error'=>'', 'Message'=>''];
    }
}
